/*----------------------------------------------------------------------------
KERNEL.C

EXPORTS
void kmain(void);
----------------------------------------------------------------------------*/
#include <_printf.h> /* do_printf() */
#include <mb_info.h> /* MBINFO_..., mboot_info_t, mboot_mod_t */
#include <string.h> /* NULL, strlen() */
#include <stdarg.h> /* va_list, va_start(), va_end() */

/* IMPORTS
from KSTART.S */
extern mboot_info_t *g_mboot_info;

/* from VIDEO.C */
void clear_screen(void);
void putch(unsigned c);

/* from VGA.C */
void set_graphics_mode(void);
void set_text_mode(void);
void display_pcx_image(unsigned char *image, unsigned long len);

/* from KBD.C */
unsigned getch(void);
/*****************************************************************************
*****************************************************************************/
static int kprintf_help(unsigned c, void **ptr)
{
	putch(c);
	return 0;
}
/*****************************************************************************
*****************************************************************************/
static void kprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	do_printf(fmt, args, kprintf_help, NULL);
	va_end(args);
}
/*****************************************************************************
*****************************************************************************/
void kmain(void)
{
	mboot_range_t *range;
	mboot_drive_t *drive;
	mboot_mod_t *mod;
	unsigned char *image;
	unsigned long len;
	unsigned i;

	clear_screen();
	kprintf("hello\n");
/* memory fields */
	if(g_mboot_info->flags & MBINFO_MEM)
		kprintf("%luK conventional memory, %luK extended memory,\n",
			g_mboot_info->conv_mem, g_mboot_info->ext_mem);
	else
		kprintf("*** Bootloader did not set memory sizes\n");
/* boot device */
	if(g_mboot_info->flags & MBINFO_BOOT_DEV)
		kprintf("Root device = 0x%X\n", g_mboot_info->boot_dev);
	else
		kprintf("*** Bootloader did not set boot device\n");
/* kernel command line */
	if(g_mboot_info->flags & MBINFO_CMD_LINE)
		kprintf("Kernel command line: '%s' (%u bytes at 0x%p)\n",
			g_mboot_info->cmd_line,
			strlen((char *)g_mboot_info->cmd_line) + 1,
			g_mboot_info->cmd_line);
	else
		kprintf("*** Bootloader did not set kernel command line\n");
/* modules */
	if(g_mboot_info->flags & MBINFO_MODS)
	{
		kprintf("%u modules (%u bytes at 0x%p):\n",
			g_mboot_info->num_mods,
			g_mboot_info->num_mods * sizeof(mboot_mod_t),
			g_mboot_info->mods_adr);
		for(i = 0; i < g_mboot_info->num_mods; i++)
		{
			mod = (mboot_mod_t *)g_mboot_info->mods_adr + i;
			kprintf("Module %2u: 0x%5lX bytes at 0x%8lX, "
				"cmdline = '%s'\n", i,
				mod->end_adr - mod->start_adr,
				mod->start_adr, mod->cmd_line);
		}
	}
	else
		kprintf("*** Bootloader did not set module fields\n");
/* BIOS memory ranges from INT 15h AX=E820h */
	if(g_mboot_info->flags & MBINFO_MEM_MAP)
	{
		kprintf("Memory map: %u bytes at 0x%p\n",
			g_mboot_info->map_len, g_mboot_info->map_adr);
		for(i = 0; i < g_mboot_info->map_len; )
		{
			range = (mboot_range_t *)(g_mboot_info->map_adr + i);
			kprintf("\tType=%u, base adr=0x%8lX, size=0x%8lX\n",
				range->type, range->adr, range->size);
			i += (4 + range->len);
		}
	}
	else
		kprintf("*** Bootloader did not set memory map fields\n");
/* drive info */
	if(g_mboot_info->flags & MBINFO_DRIVE_INFO)
	{
		kprintf("Hard drive info: %u bytes at 0x%p\n",
			g_mboot_info->drives_len, g_mboot_info->drives_adr);
		for(i = 0; i < g_mboot_info->drives_len; )
		{
			drive = (mboot_drive_t *)(g_mboot_info->drives_adr + i);
			kprintf("\tINT13h drive num=0x%02X, ", drive->drive_num);
			if(drive->use_lba)
				kprintf("LBA\n");
			else
				kprintf("CHS=%u:%u:%u\n", drive->cyls,
					drive->heads, drive->sects);
			i += drive->len;
		}
	}
	else
		kprintf("*** Bootloader did not set drive info fields\n");
/* ROM config table from INT 15h AH=C0h */
	if(g_mboot_info->flags & MBINFO_CONFIG_TABLE)
	{
		kprintf("ROM config table at 0x%X\n",
			g_mboot_info->config_table);
	}
	else
		kprintf("*** Bootloader did not set ROM config table field\n");
/* loader name */
	if(g_mboot_info->flags & MBINFO_LOADER_NAME)
	{
		kprintf("Bootloader name: '%s' (%u bytes at 0x%p)\n",
			g_mboot_info->loader_name,
			strlen((char *)g_mboot_info->loader_name) + 1,
			g_mboot_info->loader_name);
	}
	else
		kprintf("*** Bootloader did not set bootloader name field\n");
/* */
// xxx - APM and video


/* display images */
	for(i = 0; i < g_mboot_info->num_mods; i++)
	{
		mod = (mboot_mod_t *)g_mboot_info->mods_adr + i;
		len = mod->end_adr - mod->start_adr + 1;
		image = (unsigned char *)mod->start_adr;
/* validate PCX file */
		if(image[0] != 10 || image[2] != 1 ||/* PCX signature bytes */
			image[3] != 1 || /* depth (bits per pixel) */
			image[65] != 4) /* planes */
		{
			kprintf("Module %u is not a 16-color PCX file\n",
				i + 1);
			continue;
		}
		kprintf("Press a key to display image %u of %u\n",
			i + 1, g_mboot_info->num_mods);
		getch();
		set_graphics_mode();
		display_pcx_image(image, len);
		getch();
		set_text_mode();
		clear_screen();
	}
	kprintf("Done\n");
	return;
}
